<?php
require_once __DIR__ . '/../../includes/db.php';
$db = db();

// fetch full tree (including inactive for admin view)
$roots = [];
$res = $db->query("SELECT * FROM nav_items WHERE parent_id IS NULL AND deleted_at IS NULL ORDER BY sort_order ASC, id ASC");
while($r = $res->fetch_assoc()){
  $r['children'] = [];
  $stmt = $db->prepare("SELECT * FROM nav_items WHERE parent_id=? AND deleted_at IS NULL ORDER BY sort_order ASC, id ASC");
  $stmt->bind_param("i", $r['id']);
  $stmt->execute();
  $r2 = $stmt->get_result();
  while($ch = $r2->fetch_assoc()){
    $r['children'][] = $ch;
  }
  $roots[] = $r;
}
?>

<div class="d-flex flex-wrap align-items-center justify-content-between gap-2 mb-3">
  <div>
    <div class="fw-semibold">Navbar Modül Yönetimi</div>
    <div class="text-muted small">Ekle / Düzenle / Kopyala / Sil • Sürükle-bırak sıralama</div>
  </div>
  <button class="btn btn-primary" id="btnAddRoot"><i class="bi bi-plus-lg me-1"></i> Navbar Modül Ekle</button>
</div>

<div class="card p-3">
  <div class="table-responsive">
    <table class="table table-sm" id="navTable">
      <thead class="table-light">
        <tr>
          <th style="width:36px;"></th>
          <th>Başlık</th>
          <th>Key</th>
          <th>Tip</th>
          <th>URL</th>
          <th>Aktif</th>
          <th>Rol</th>
          <th>Permission</th>
          <th class="text-end" style="width:220px;">İşlemler</th>
        </tr>
      </thead>
      <tbody>
        <?php foreach($roots as $it): ?>
          <tr data-id="<?= (int)$it['id'] ?>" data-parent="0">
            <td class="drag-handle text-muted"><i class="bi bi-grip-vertical"></i></td>
            <td class="fw-semibold">
              <?php if(!empty($it['icon'])): ?><i class="bi <?= htmlspecialchars($it['icon']) ?> me-2"></i><?php endif; ?>
              <?= htmlspecialchars($it['title']) ?>
            </td>
            <td class="text-muted"><?= htmlspecialchars($it['key']) ?></td>
            <td><span class="badge text-bg-light border"><?= htmlspecialchars($it['type']) ?></span></td>
            <td class="text-muted small"><?= htmlspecialchars($it['url'] ?? '') ?></td>
            <td><?= ((int)$it['is_active']===1) ? '<span class="badge text-bg-success">aktif</span>' : '<span class="badge text-bg-secondary">pasif</span>' ?></td>
            <td class="text-muted small"><?= htmlspecialchars($it['role_code'] ?? '') ?></td>
            <td class="text-muted small"><?= htmlspecialchars($it['permission_code'] ?? '') ?></td>
            <td class="text-end">
              <button class="btn btn-sm btn-outline-secondary" onclick="openEdit(<?= (int)$it['id'] ?>)"><i class="bi bi-pencil"></i> Düzenle</button>
              <button class="btn btn-sm btn-outline-primary" onclick="copyItem(<?= (int)$it['id'] ?>)"><i class="bi bi-copy"></i></button>
              <button class="btn btn-sm btn-outline-danger" onclick="deleteItem(<?= (int)$it['id'] ?>)"><i class="bi bi-trash"></i></button>
            </td>
          </tr>

          <?php foreach(($it['children'] ?? []) as $ch): ?>
            <tr data-id="<?= (int)$ch['id'] ?>" data-parent="<?= (int)$it['id'] ?>">
              <td class="text-muted"><i class="bi bi-dot"></i></td>
              <td class="ps-4">
                <?php if(!empty($ch['icon'])): ?><i class="bi <?= htmlspecialchars($ch['icon']) ?> me-2"></i><?php endif; ?>
                <?= htmlspecialchars($ch['title']) ?>
              </td>
              <td class="text-muted"><?= htmlspecialchars($ch['key']) ?></td>
              <td><span class="badge text-bg-light border"><?= htmlspecialchars($ch['type']) ?></span></td>
              <td class="text-muted small"><?= htmlspecialchars($ch['url'] ?? '') ?></td>
              <td><?= ((int)$ch['is_active']===1) ? '<span class="badge text-bg-success">aktif</span>' : '<span class="badge text-bg-secondary">pasif</span>' ?></td>
              <td class="text-muted small"><?= htmlspecialchars($ch['role_code'] ?? '') ?></td>
              <td class="text-muted small"><?= htmlspecialchars($ch['permission_code'] ?? '') ?></td>
              <td class="text-end">
                <button class="btn btn-sm btn-outline-secondary" onclick="openEdit(<?= (int)$ch['id'] ?>)"><i class="bi bi-pencil"></i> Düzenle</button>
                <button class="btn btn-sm btn-outline-primary" onclick="copyItem(<?= (int)$ch['id'] ?>)"><i class="bi bi-copy"></i></button>
                <button class="btn btn-sm btn-outline-danger" onclick="deleteItem(<?= (int)$ch['id'] ?>)"><i class="bi bi-trash"></i></button>
              </td>
            </tr>
          <?php endforeach; ?>

        <?php endforeach; ?>
      </tbody>
    </table>
  </div>

  <div class="text-muted small mt-2">
    Not: Root menü sırası sürükle-bırak ile değişir. (Child sıralama bu demo sürümde manuel <i>sort_order</i> ile.)
  </div>
</div>

<!-- Modal -->
<div class="modal fade" id="navModal" tabindex="-1">
  <div class="modal-dialog modal-lg">
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title" id="navModalTitle">Navbar Modül</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
      </div>
      <div class="modal-body">
        <form id="navForm">
          <input type="hidden" name="id" value="0">
          <input type="hidden" name="parent_id" value="">

          <div class="row g-3">
            <div class="col-12 col-md-6">
              <label class="form-label">Başlık</label>
              <input class="form-control" name="title" required>
            </div>
            <div class="col-12 col-md-6">
              <label class="form-label">Key (benzersiz)</label>
              <input class="form-control" name="key" required>
            </div>
            <div class="col-12 col-md-4">
              <label class="form-label">İkon (Bootstrap Icons)</label>
              <input class="form-control" name="icon" placeholder="bi-house">
            </div>
            <div class="col-12 col-md-4">
              <label class="form-label">Tip</label>
              <select class="form-select" name="type">
                <option value="link">link</option>
                <option value="dropdown">dropdown</option>
              </select>
            </div>
            <div class="col-12 col-md-4">
              <label class="form-label">Aktif</label>
              <select class="form-select" name="is_active">
                <option value="1">aktif</option>
                <option value="0">pasif</option>
              </select>
            </div>
            <div class="col-12">
              <label class="form-label">URL (link ise)</label>
              <input class="form-control" name="url" placeholder="/home.php">
            </div>

            <div class="col-12 col-md-4">
              <label class="form-label">Sıra</label>
              <input class="form-control" name="sort_order" type="number" value="100">
            </div>
            <div class="col-12 col-md-4">
              <label class="form-label">Rol (opsiyonel)</label>
              <input class="form-control" name="role_code" placeholder="admin / staff / customer">
            </div>
            <div class="col-12 col-md-4">
              <label class="form-label">Permission (opsiyonel)</label>
              <input class="form-control" name="permission_code" placeholder="settings.manage">
            </div>
          </div>
        </form>

        <div class="alert alert-info small mt-3 mb-0">
          Dropdown ise URL boş kalabilir. Child menü eklemek için önce parent kaydını oluşturup sonra child ekleyebilirsin (parent_id ile).
        </div>
      </div>
      <div class="modal-footer">
        <button class="btn btn-outline-secondary" data-bs-dismiss="modal">Vazgeç</button>
        <button class="btn btn-primary" id="btnSaveNav"><i class="bi bi-save me-1"></i> Kaydet</button>
      </div>
    </div>
  </div>
</div>

<script>
const modalEl = document.getElementById('navModal');
const navModal = new bootstrap.Modal(modalEl);

document.getElementById('btnAddRoot').addEventListener('click', ()=>{
  openNew(0);
});

function openNew(parentId){
  document.getElementById('navModalTitle').innerText = parentId ? 'Alt Menü Ekle' : 'Navbar Modül Ekle';
  const f = document.getElementById('navForm');
  f.reset();
  f.id.value = 0;
  f.parent_id.value = parentId || '';
  f.sort_order.value = 100;
  f.is_active.value = 1;
  f.type.value = 'link';
  navModal.show();
}

async function openEdit(id){
  const r = await fetch('/api/navbar_actions.php?action=get&id='+encodeURIComponent(id));
  const j = await r.json();
  if(!j.ok){ toast(j.msg||'Hata','danger'); return; }
  const it = j.item;
  document.getElementById('navModalTitle').innerText = 'Navbar Modül Düzenle';
  const f = document.getElementById('navForm');
  f.id.value = it.id;
  f.parent_id.value = it.parent_id || '';
  f.title.value = it.title || '';
  f.key.value = it.key || '';
  f.icon.value = it.icon || '';
  f.type.value = it.type || 'link';
  f.url.value = it.url || '';
  f.sort_order.value = it.sort_order || 100;
  f.is_active.value = it.is_active ? 1 : 0;
  f.role_code.value = it.role_code || '';
  f.permission_code.value = it.permission_code || '';
  navModal.show();
}

document.getElementById('btnSaveNav').addEventListener('click', async ()=>{
  const f = document.getElementById('navForm');
  const data = Object.fromEntries(new FormData(f).entries());
  data.id = parseInt(data.id||'0',10);
  data.parent_id = data.parent_id ? parseInt(data.parent_id,10) : null;
  data.sort_order = parseInt(data.sort_order||'100',10);
  data.is_active = parseInt(data.is_active||'1',10);
  try{
    const j = await postJSON('/api/navbar_actions.php?action=save', data);
    if(j.ok){ toast('Kaydedildi','success'); location.reload(); }
    else toast(j.msg||'Hata','danger');
  }catch(e){
    toast(e.message||'Hata','danger');
  }
});

async function copyItem(id){
  if(!confirm('Bu navbar modülünü kopyalayalım mı?')) return;
  try{
    const j = await postJSON('/api/navbar_actions.php?action=copy', {id});
    if(j.ok){ toast('Kopyalandı','success'); location.reload(); }
    else toast(j.msg||'Hata','danger');
  }catch(e){ toast(e.message,'danger'); }
}

async function deleteItem(id){
  if(!confirm('Silinsin mi? (soft delete)')) return;
  try{
    const j = await postJSON('/api/navbar_actions.php?action=delete', {id});
    if(j.ok){ toast('Silindi','success'); location.reload(); }
    else toast(j.msg||'Hata','danger');
  }catch(e){ toast(e.message,'danger'); }
}

/* Root drag-drop ordering (simple) */
const tbody = document.querySelector('#navTable tbody');
let dragRow = null;

tbody.addEventListener('dragstart', (e)=>{
  const tr = e.target.closest('tr[data-parent="0"]');
  if(!tr) return;
  dragRow = tr;
  e.dataTransfer.effectAllowed = 'move';
});
tbody.addEventListener('dragover', (e)=>{
  if(!dragRow) return;
  e.preventDefault();
  const over = e.target.closest('tr[data-parent="0"]');
  if(!over || over===dragRow) return;
  const rect = over.getBoundingClientRect();
  const before = (e.clientY - rect.top) < rect.height/2;
  tbody.insertBefore(dragRow, before ? over : over.nextSibling);
});
tbody.addEventListener('drop', async (e)=>{
  if(!dragRow) return;
  e.preventDefault();
  dragRow = null;

  // build new order from root rows only
  const ids = Array.from(tbody.querySelectorAll('tr[data-parent="0"]')).map(tr=>parseInt(tr.dataset.id,10));
  try{
    const j = await postJSON('/api/navbar_actions.php?action=reorder_roots', {ids});
    if(j.ok) toast('Sıra güncellendi','success');
    else toast(j.msg||'Hata','danger');
  }catch(err){ toast(err.message,'danger'); }
});
/* make root rows draggable */
Array.from(tbody.querySelectorAll('tr[data-parent="0"]')).forEach(tr=>{
  tr.setAttribute('draggable','true');
});
</script>
